
#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.DAL.UnitTests
{
	[TestFixture]
	public class DAL_Shipment : BaseTest 
	{
		
		private DataTable _dtShipment;
		private DataTable _dtBloodUnit;
		private DataTable _dtBloodUnitStatus;
		private DateTime _currentDateTime;

		private string _baseQuery = string.Concat(	"SELECT TOP 1 * FROM Shipment WHERE ShipmentDirection = 1 ",
			"AND ProductShipperGuid IS NOT NULL ",
			"AND RecordStatusCode = 'A' ",
			"AND DivisionCode = '", UnitTestConfig.TestDivisionCode, "'" );

		//
		private string _bloodUnitQuery = string.Concat(	"SELECT TOP 1 BU.*, BUSCC.UnitStatusCode ",
			"FROM vw_BloodShipAllCurrent BU ",
			"JOIN BloodUnitStatus BUS ON BUS.BloodUnitGuid = BU.BloodUnitGuid ",
			"JOIN BloodUnitStatusCodeCurrent BUSCC ON BUSCC.BloodUnitGuid = BU.BloodUnitGuid ",
			"WHERE OutgoingShipmentGuid IN (SELECT ShipmentGuid FROM Shipment WHERE ShipmentDate IS NOT NULL) ",
			"AND BU.RecordStatusCode = 'A'" ,
			"AND BU.DivisionCode = '", UnitTestConfig.TestDivisionCode, "'" );

		[SetUp]
		protected void SetUp() 
		{				
			this.BaseSetUp();

			if (this.RefreshData)
			{
				this._dtShipment = UnitTestUtil.RunSQLGetDataTable(_baseQuery, false);
				if (this._dtShipment.Rows.Count == 0)
				{
					DataCreator.CreateBloodUnit(true);
					this._dtShipment = UnitTestUtil.RunSQLGetDataTable(_baseQuery, true);
				}
				//*BloodUnit table (outgoing shipment)
				this._dtBloodUnit = UnitTestUtil.RunSQLGetDataTable(_bloodUnitQuery, false);
				
				if (this._dtBloodUnit.Rows.Count == 0)
				{
					BOL.BloodUnit tmpXUnit = DataCreator.CreateBloodUnit(true);
					string invoiceNumber = new Random().Next(111111111,999999999).ToString();
					BOL.LocalSupplier tmpLS = DataCreator.CreateLocalSupplier();
					//
					BOL.OutgoingShipmentCollection tmpOSC = BOL.OutgoingShipmentCollection.CreateOutgoingInvoice(
						invoiceNumber, tmpLS.LocalSupplierGuid, DateTime.Now, Common.ShipmentType.ReturnToSupplier, 
						Common.LogonUser.LogonUserName, Common.UpdateFunction.UnitTests);
					//
					BOL.OutgoingShipmentCollection.LineItem lineItem = 
						BOL.OutgoingShipmentCollection.LineItem.GetLineItemForUpdate(tmpXUnit.BloodUnitGuid);
					//
					tmpOSC.AddUnitToInvoice(lineItem);
					tmpOSC.SaveChangesToInvoiceItems(new ArrayList(), Common.UpdateFunction.UnitTests);			//unit is not X, confirmation pending 
					DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.UnitLogout);
					tmpOSC.ConfirmShipment(
						DateTime.Now, Common.UpdateFunction.UnitTests, Common.WorkloadProcessID.UnitLogout);	//This puts the unit in X
					//
					this._dtBloodUnit = UnitTestUtil.RunSQLGetDataTable(_bloodUnitQuery, true);
				}

				
				//*BloodUnitStatus table, always true, OR ELSE WE'RE DOOOOOOOOOOOOOOMED!
				this._dtBloodUnitStatus = UnitTestUtil.RunSQLGetDataTable(
					string.Concat( "SELECT * FROM BloodUnitStatus WHERE BloodUnitGuid = '", 
					this._dtBloodUnit.Rows[0][TABLES.BloodUnit.BloodUnitGuid], "'" ), true);
				//

				this.RefreshData = false;

			}

			this._currentDateTime = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
		}

		[Test]
		public void GetIncomingShipment_String_Guid_Pass()
		{
			Guid unitGuid = (Guid)this._dtShipment.Rows[0][TABLES.Shipment.LocalSupplierGuid];
			string invoiceNumber = this._dtShipment.Rows[0][TABLES.Shipment.InvoiceNumber].ToString();

			DataTable dtTest = DAL.Shipment.GetIncomingShipment(invoiceNumber, unitGuid);

			Assert.AreEqual(true, dtTest.Rows.Count > 0, "1-Rows.Count");
            Assert.AreEqual(6, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentGuid), TABLES.Shipment.ShipmentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDate), TABLES.Shipment.ShipmentDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ProductShipperGuid), TABLES.Shipment.ProductShipperGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.InvoiceNumber), TABLES.Shipment.InvoiceNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.RowVersion), TABLES.Shipment.RowVersion);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.LocalSupplierGuid), TABLES.Shipment.LocalSupplierGuid);

			//Check without the supplierGuid
			dtTest = DAL.Shipment.GetIncomingShipment(invoiceNumber, Guid.Empty);
			Assert.AreEqual(true, dtTest.Rows.Count > 0, "2-Rows.Count");
		}

		[Test]
		public void GetIncomingShipment_String_Guid_Fail()
		{
			DataTable dtTest = DAL.Shipment.GetIncomingShipment("Flexo", Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetIncomingShipment_String_Guid_Date_Pass()
		{
			Guid unitGuid = (Guid)this._dtShipment.Rows[0][TABLES.Shipment.LocalSupplierGuid];
			string invoiceNumber = this._dtShipment.Rows[0][TABLES.Shipment.InvoiceNumber].ToString();
			System.DateTime processingDate = (System.DateTime)this._dtShipment.Rows[0][TABLES.Shipment.ShipmentDate];

			DataTable dtTest = DAL.Shipment.GetIncomingShipment(invoiceNumber, unitGuid, processingDate);

			Assert.AreEqual(6, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentGuid), TABLES.Shipment.ShipmentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDate), TABLES.Shipment.ShipmentDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ProductShipperGuid), TABLES.Shipment.ProductShipperGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.InvoiceNumber), TABLES.Shipment.InvoiceNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.RowVersion), TABLES.Shipment.RowVersion);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.LocalSupplierGuid), TABLES.Shipment.LocalSupplierGuid);
		}

		[Test]
		public void GetIncomingShipment_String_Guid_Date_Fail()
		{
			DataTable dtTest = DAL.Shipment.GetIncomingShipment("Bender", Guid.NewGuid(), System.DateTime.Now);

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetOutgoingBloodUnitShipment_Pass()
		{
			Guid unitGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid FROM BloodUnitShipment WHERE RecordStatusCode = 'A' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM BloodUnit WHERE RecordStatusCode = 'A' AND DivisionCode = '" + this.TestDivisionCode + "')");
			DataTable dtTest = DAL.Shipment.GetOutgoingBloodUnitShipment(unitGuid);

			Assert.AreEqual(6, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitShipment.BloodUnitShipmentGuid), TABLES.BloodUnitShipment.BloodUnitShipmentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitShipment.BloodUnitGuid), TABLES.BloodUnitShipment.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitShipment.OutgoingShipmentGuid), TABLES.BloodUnitShipment.OutgoingShipmentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDate), TABLES.Shipment.ShipmentDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitShipment.RecordStatusCode), TABLES.BloodUnitShipment.RecordStatusCode);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitShipment.RowVersion), TABLES.BloodUnitShipment.RowVersion);
		}

		[Test]
		public void GetOutgoingBloodUnitShipment_Fail()
		{
			DataTable dtTest = DAL.Shipment.GetOutgoingBloodUnitShipment(Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		
		[Test]
		public void GetOriginalBloodUnitShipper_Pass()
		{
			Guid unittestGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid FROM BloodUnitShipment WHERE RecordStatusCode = 'A' AND BloodUnitGuid IN (SELECT BloodUnitGuid FROM BloodUnit WHERE RecordStatusCode = 'A' AND OutgoingShipmentGuid IS NOT NULL AND DivisionCode = '" + this.TestDivisionCode + "')");
			System.Data.DataTable dt = DAL.Shipment.GetOriginalBloodUnitShipper(unittestGuid);
			System.Guid localSupplierGuid = System.Guid.Empty;
			string localFacilityName = string.Empty;
			if(dt.Rows.Count >0)
			{
				localSupplierGuid = (System.Guid)dt.Rows[0][Common.VbecsTables.LocalSupplier.LocalSupplierGuid];
				localFacilityName = dt.Rows[0][Common.VbecsTables.LocalSupplier.FacilityName].ToString();
			}
			Assert.AreEqual(true, dt.Columns.Contains(TABLES.LocalSupplier.LocalSupplierGuid), TABLES.LocalSupplier.LocalSupplierGuid);
			Assert.AreEqual(true, dt.Columns.Contains(TABLES.LocalSupplier.FacilityGuid), TABLES.LocalSupplier.FacilityGuid);
			Assert.AreEqual(true, dt.Columns.Contains(TABLES.LocalSupplier.FacilityName), TABLES.LocalSupplier.FacilityName);
			Assert.AreNotEqual(localSupplierGuid, System.Guid.Empty);
		}

		[Test]
		public void GetOriginalBloodUnitShipper_Fail()
		{
			DataTable dtTest = DAL.Shipment.GetOriginalBloodUnitShipper(Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetShipmentsWithAvailableUnitsPastFiveDays_Pass()
		{
			DataTable dtTest = DAL.Shipment.GetShipmentsWithAvailableUnitsPastFiveDays();

			Assert.AreEqual(6, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentGuid), TABLES.Shipment.ShipmentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.InvoiceNumber), TABLES.Shipment.InvoiceNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDate), TABLES.Shipment.ShipmentDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FacilityName), TABLES.LocalSupplier.FacilityName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.LocalSupplierGuid), TABLES.Shipment.LocalSupplierGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDirection), TABLES.Shipment.ShipmentDirection);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetShipmentsWithAvailableUnitsPastFiveDays_Fail()
		{
		}

		[Test]
		public void GetShipment_Pass()
		{
			Guid unitGuid = (Guid)this._dtShipment.Rows[0][TABLES.Shipment.ShipmentGuid];

			DataTable dtTest = DAL.Shipment.GetShipment(unitGuid);

			Assert.AreEqual(true, dtTest.Columns.Count == 12, "Columns.Count");
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.LocalSupplierGuid), TABLES.Shipment.LocalSupplierGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ProductShipperGuid), TABLES.Shipment.ProductShipperGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.InvoiceNumber), TABLES.Shipment.InvoiceNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDate), TABLES.Shipment.ShipmentDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDirection), TABLES.Shipment.ShipmentDirection);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentGuid), TABLES.Shipment.ShipmentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentTypeCode), TABLES.Shipment.ShipmentTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentTemperature), TABLES.Shipment.ShipmentTemperature);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.DivisionCode), TABLES.Shipment.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.LastUpdateDate), TABLES.Shipment.LastUpdateDate);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.LastUpdateUser), TABLES.Shipment.LastUpdateUser);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.RowVersion), TABLES.Shipment.RowVersion);
		}

		[Test]
		public void GetShipment_Fail()
		{
			DataTable dtTest = DAL.Shipment.GetShipment(Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetShipmentByUnit_Pass()
		{
			Guid unitGuid = (Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnit.BloodUnitGuid];

			DataTable dtTest = DAL.Shipment.GetShipmentByUnit(unitGuid);

			Assert.AreEqual(true, dtTest.Rows.Count > 0, "Rows.Count");
			Assert.AreEqual(11, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.LocalSupplierGuid), TABLES.Shipment.LocalSupplierGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.InvoiceNumber), TABLES.Shipment.InvoiceNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDate), TABLES.Shipment.ShipmentDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDirection), TABLES.Shipment.ShipmentDirection);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentGuid), TABLES.Shipment.ShipmentGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentTypeCode), TABLES.Shipment.ShipmentTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentTemperature), TABLES.Shipment.ShipmentTemperature);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.DivisionCode), TABLES.Shipment.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.LastUpdateDate), TABLES.Shipment.LastUpdateDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.LastUpdateUser), TABLES.Shipment.LastUpdateUser);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.RowVersion), TABLES.Shipment.RowVersion);
		}

		[Test]
		public void GetShipmentByUnit_Fail()
		{
			DataTable dtTest = DAL.Shipment.GetShipmentByUnit(Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetIncomingShipments_Pass()
		{
			DataTable dtTest = DAL.Shipment.GetIncomingShipments(_currentDateTime.AddYears(-1), _currentDateTime);
			
			Assert.AreEqual(13, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentGuid), TABLES.Shipment.ShipmentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.InvoiceNumber), TABLES.Shipment.InvoiceNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDate), TABLES.Shipment.ShipmentDate);
			Assert.AreEqual(true , dtTest.Columns.Contains(TABLES.BloodUnitShipment.BloodUnitGuid), TABLES.BloodUnitShipment.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FacilityName), TABLES.LocalSupplier.FacilityName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.EyeReadableUnitId), TABLES.BloodUnit.EyeReadableUnitId);
			Assert.AreEqual(true , dtTest.Columns.Contains(TABLES.DonationType.DonationTypeText), TABLES.DonationType.DonationTypeText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.BloodTypeCode), TABLES.BloodUnitMedia.BloodTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RhFactor.RhFactorText), TABLES.RhFactor.RhFactorText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitFinance.UnitCost), TABLES.BloodUnitFinance.UnitCost);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitFinance.SpecialTestingCost), TABLES.BloodUnitFinance.SpecialTestingCost);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.UnitStatus.UnitStatusText), TABLES.UnitStatus.UnitStatusText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductShortName), TABLES.BloodProduct.ProductShortName);
		}

		[Test]
		public void GetIncomingShipments_Fail()
		{
			DataTable dtTest = DAL.Shipment.GetIncomingShipments(_currentDateTime.AddDays(1), _currentDateTime);
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}


		[Test]
		public void GetOutgoingShipments_Pass()
		{
			DataTable dtTest = DAL.Shipment.GetOutgoingShipments(_currentDateTime.AddYears(-1), _currentDateTime);
			
			Assert.AreEqual(9, dtTest.Columns.Count, "Columns.Count");
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentGuid), TABLES.Shipment.ShipmentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.InvoiceNumber), TABLES.Shipment.InvoiceNumber);
			Assert.AreEqual(true , dtTest.Columns.Contains(TABLES.BloodUnitShipment.BloodUnitGuid), TABLES.BloodUnitShipment.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Shipment.ShipmentDate), TABLES.Shipment.ShipmentDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.LocalSupplier.FacilityName), TABLES.LocalSupplier.FacilityName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.EyeReadableUnitId), TABLES.BloodUnit.EyeReadableUnitId);
			Assert.AreEqual(true , dtTest.Columns.Contains(TABLES.DonationType.DonationTypeText), TABLES.DonationType.DonationTypeText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitFinance.ReturnCreditAmount), TABLES.BloodUnitFinance.ReturnCreditAmount);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductShortName), TABLES.BloodProduct.ProductShortName);
		}

		[Test]
		public void GetOutgoingShipments_Fail()
		{
			DataTable dtOutgoingShipments = DAL.Shipment.GetOutgoingShipments(_currentDateTime.AddDays(1), _currentDateTime);
			Assert.AreEqual(0, dtOutgoingShipments.Rows.Count, "Rows.Count");
		}


		[Test]
		public void SaveIncomingShipment_Pass()
		{
			this.RefreshData = true;

			Guid unitGuid = Guid.NewGuid();
			Guid bpGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodProductGuid FROM BloodProduct WHERE ProductCode NOT LIKE 'E%'");
			Guid shipmentGuid = Guid.NewGuid();

			BOL.BloodProduct tstBP = new BOL.BloodProduct(bpGuid);
			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid, bpGuid);
			BOL.Division tmpDiv = new BOL.Division(this.TestDivisionCode);
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(tmpDiv.LocalSupplierGuid, false);

			//Shipment now requires ProductShipperGuid, so, we're going to create ProductShipper(if not one already)
			BOL.ProductShipper tmpPS = tstBP.ProductShippers.GetElementByLocalSupplierGuid(tmpLS.LocalSupplierGuid);
			if (tmpPS == null)
			{
				tmpPS = new BOL.ProductShipper(Guid.NewGuid(), bpGuid);
				tmpPS.ProductCost = 1;
				tmpPS.LocalSupplierGuid = tmpLS.LocalSupplierGuid;
				tmpPS.ReturnCreditPercentage = 1;
				tmpPS.RecordStatusCode = Common.RecordStatusCode.Active;
				tstBP.ProductShippers.Add(tmpPS);
				tstBP.Save(Common.UpdateFunction.UnitTests);
			}

			//Set our BloodUnit properties
			tmpUnit.LabelTypeCode = "C";
			tmpUnit.UnitFdaRegistrationNumber = tmpLS.FDARegistrationNumber;

			string unitId = new Random().Next(11111,99999).ToString() + new Random().Next(11111,99999).ToString();
			tmpUnit.SetLabelUnitId(unitId, false);
			tmpUnit.SetEyeReadableUnitId(unitId, false);
			
			tmpUnit.BloodProductGuid = bpGuid;
			tmpUnit.DonationType = new BOL.DonationType(2);
			tmpUnit.OriginalVolume = 420;

			tmpUnit.EntryMethodCode = "M";
			tmpUnit.RecordStatusCode = Common.RecordStatusCode.Active;
			tmpUnit.BloodUnitFinance.UnitCost = 98.76M;
			tmpUnit.BloodUnitFinance.SpecialTestingCost = 54.32M;
			tmpUnit.BloodUnitMedia.UnitExpirationDate = System.DateTime.Now.AddDays(7);
			tmpUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			tmpUnit.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			tmpUnit.Comments = "Unit Tests - 1411 - New Unit";

			DataTable dtIShipment = SprocHelper.GetInsertShipmentSprocTable(shipmentGuid, tmpDiv.LocalSupplierGuid, tmpPS.ProductShipperGuid);
			DataTable dtIBUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(false);
			DataTable dtIBUnitFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(false);
			DataTable dtIBUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(false);
			DataTable dtIBUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(false);
			DataTable dtIBUnitShipment = DAL.BloodUnitModification.GetEmptyBloodUnitShipmentTable();

			dtIBUnit.Rows.Add(tmpUnit.LoadDataRowFromThis(dtIBUnit.NewRow()));
			dtIBUnitFinance.Rows.Add(tmpUnit.BloodUnitFinance.LoadDataRowFromThis(dtIBUnitFinance.NewRow()));
			dtIBUnitMedia.Rows.Add(tmpUnit.BloodUnitMedia.LoadDataRowFromThis(dtIBUnitMedia.NewRow()));
			dtIBUnitStatus.Rows.Add(tmpUnit.BloodUnitStatus.LoadDataRowFromThis(dtIBUnitStatus.NewRow()));

			//Build the shipment row
			DataRow drBUS = dtIBUnitShipment.NewRow();
			drBUS[TABLES.BloodUnitShipment.BloodUnitShipmentGuid] = Guid.NewGuid();
			drBUS[TABLES.BloodUnitShipment.BloodUnitGuid] = unitGuid;
			drBUS[TABLES.BloodUnitShipment.IncomingShipmentGuid] = shipmentGuid;
			drBUS[TABLES.BloodUnitShipment.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drBUS[TABLES.BloodUnitShipment.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			dtIBUnitShipment.Rows.Add(drBUS);

			DataSet dsSave = new DataSet();
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIShipment, Common.UpdateFunction.UnitTests));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIBUnit, Common.UpdateFunction.UnitTests));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIBUnitFinance, Common.UpdateFunction.UnitTests));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIBUnitMedia, Common.UpdateFunction.UnitTests));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIBUnitStatus, Common.UpdateFunction.UnitTests));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIBUnitShipment, Common.UpdateFunction.UnitTests));
			
			//Add an antigen for code coverage
			DataTable dtAntigen = new DataTable(TABLES.BloodUnitAntigen.TableName);
			dtAntigen = SprocHelper.GetInsertBloodUnitAntigenSprocTable(unitGuid);
			dsSave.Tables.Add(dtAntigen);

			DAL.Shipment.SaveIncomingShipment(shipmentGuid, false, dsSave, Common.UpdateFunction.UnitTests);

			Assert.IsTrue(true, "New Unit-Save");

			//`````````````````````````````````````````````````````````````````````````````````````
			//Save a existing unit
			//.....................................................................................
			DataTable dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE UnitStatusCode = 'A' AND DivisionCode = '" + this.TestDivisionCode + "' AND  BloodUnitGuid IN (SELECT BloodUnitGuid FROM BloodUnitAntigen)", false);
			if (dataCheck.Rows.Count == 0)
			{
				BOL.BloodUnit buaUnit = DataCreator.CreateRBCBloodUnit(false);
				DataCreator.CreateAboRhConfirmationTestsForUnit(buaUnit.BloodUnitGuid,buaUnit.BloodUnitMedia.AboRh);	//UnitStatus = A
				DataTable insertBUA = SprocHelper.GetInsertBloodUnitAntigenSprocTable(buaUnit.BloodUnitGuid);
				new Common.StoredProcedure().TransactionalGetValue(SPROCS.InsertBloodUnitAntigen.StoredProcName, insertBUA);
			}
			unitGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE UnitStatusCode = 'A' AND DivisionCode = '" + this.TestDivisionCode + "' AND  BloodUnitGuid IN (SELECT BloodUnitGuid FROM BloodUnitAntigen)");
            bpGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodProductGuid FROM vw_BloodShipAllCurrent WHERE BloodUnitGuid = '" + unitGuid.ToString() + "'");
			shipmentGuid = DataCreator.GetRecordGuid("SELECT TOP 1 IncomingShipmentGuid FROM vw_BloodShipAllCurrent WHERE BloodUnitGuid = '" + unitGuid.ToString() + "'");

			tstBP = new BOL.BloodProduct(bpGuid);
			tmpUnit = new BOL.BloodUnit(unitGuid);
			tmpUnit.Comments = "Unit Tests - 1411 - Existing Unit";

			DataTable dtUBUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(true);
			DataTable dtUBUnitFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(true);
			DataTable dtUBUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(true);
			DataTable dtUBUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);
			//.NET barks, since dtException is in another dataset already
			DataTable dtException = DAL.ExceptionReport.GetExceptionReportExpiredUnitReceived(unitGuid).Tables[0].Copy();

			dtUBUnit.Rows.Add(tmpUnit.LoadDataRowFromThis(dtUBUnit.NewRow()));
			dtUBUnitFinance.Rows.Add(tmpUnit.BloodUnitFinance.LoadDataRowFromThis(dtUBUnitFinance.NewRow()));
			dtUBUnitMedia.Rows.Add(tmpUnit.BloodUnitMedia.LoadDataRowFromThis(dtUBUnitMedia.NewRow()));
			dtUBUnitStatus.Rows.Add(tmpUnit.BloodUnitStatus.LoadDataRowFromThis(dtUBUnitStatus.NewRow()));

			DataSet dsUpdate = new DataSet();
			dsUpdate.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtUBUnit, Common.UpdateFunction.UnitTests));
			dsUpdate.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtUBUnitFinance, Common.UpdateFunction.UnitTests));
			dsUpdate.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtUBUnitMedia, Common.UpdateFunction.UnitTests));
			dsUpdate.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtUBUnitStatus, Common.UpdateFunction.UnitTests));
			dsUpdate.Tables.Add(dtException);

			//Add an antigen for code coveratge
			dtAntigen = new DataTable(TABLES.BloodUnitAntigen.TableName);
			dtAntigen = SprocHelper.GetInsertBloodUnitAntigenSprocTable(unitGuid);
			dtAntigen.Rows[0][TABLES.BloodUnitAntigen.AntigenTypeId] = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 AntigenTypeId FROM BloodUnitAntigen WHERE BloodUnitGuid = '" + unitGuid.ToString() + "'")[TABLES.BloodUnitAntigen.AntigenTypeId];
			dsUpdate.Tables.Add(dtAntigen);

			DAL.Shipment.SaveIncomingShipment(shipmentGuid, true, dsUpdate, Common.UpdateFunction.UnitTests);
			Assert.IsTrue(true, "Existing Unit-Save");
		}

		[Test]
		[ExpectedException(typeof(Common.UniqueDatabaseConstraintViolatedException))]
		public void SaveIncomingShipment_Fail()
		{
			this.RefreshData = true;

			Guid unitGuid = Guid.NewGuid();
			DataRow drProduct = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM BloodProduct WHERE ProductTypeCode = 'E002' AND RecordStatusCode = 'A' ORDER BY LastUpdateDate DESC");
			Guid bloodProductGuid = (Guid)drProduct[TABLES.BloodProduct.BloodProductGuid];
			bool isCodabar = (drProduct[TABLES.BloodProduct.ProductCodeIndicator].ToString() == "0");
			DataCreator.CreateBloodUnit(unitGuid,Guid.NewGuid(),Guid.Empty,bloodProductGuid, isCodabar);
			//
			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid);
			Guid shipmentGuid = tmpUnit.IncomingShipment.ShipmentGuid;

			DataTable dtIShipment = SprocHelper.GetInsertShipmentSprocTable(shipmentGuid, tmpUnit.IncomingShipment.LocalSupplierGuid , tmpUnit.IncomingShipment.ProductShipperGuid);
			DataTable dtIBUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(false);
			DataTable dtIBUnitFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(false);
			DataTable dtIBUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(false);
			DataTable dtIBUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(false);
			DataTable dtIBUnitShipment = DAL.BloodUnitModification.GetEmptyBloodUnitShipmentTable();

			dtIBUnit.Rows.Add(tmpUnit.LoadDataRowFromThis(dtIBUnit.NewRow()));
			dtIBUnitFinance.Rows.Add(tmpUnit.BloodUnitFinance.LoadDataRowFromThis(dtIBUnitFinance.NewRow()));
			dtIBUnitMedia.Rows.Add(tmpUnit.BloodUnitMedia.LoadDataRowFromThis(dtIBUnitMedia.NewRow()));
			dtIBUnitStatus.Rows.Add(tmpUnit.BloodUnitStatus.LoadDataRowFromThis(dtIBUnitStatus.NewRow()));

			//Build the shipment row
			DataRow drBUS = dtIBUnitShipment.NewRow();
			drBUS[TABLES.BloodUnitShipment.BloodUnitShipmentGuid] = Guid.NewGuid();
			drBUS[TABLES.BloodUnitShipment.BloodUnitGuid] = unitGuid;
			drBUS[TABLES.BloodUnitShipment.IncomingShipmentGuid] = shipmentGuid;
			drBUS[TABLES.BloodUnitShipment.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drBUS[TABLES.BloodUnitShipment.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			dtIBUnitShipment.Rows.Add(drBUS);

			DataSet dsSave = new DataSet();
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIShipment, Common.UpdateFunction.UnitTests));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIBUnit, Common.UpdateFunction.UnitTests));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIBUnitFinance, Common.UpdateFunction.UnitTests));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIBUnitMedia, Common.UpdateFunction.UnitTests));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIBUnitStatus, Common.UpdateFunction.UnitTests));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtIBUnitShipment, Common.UpdateFunction.UnitTests));
			
			DAL.Shipment.SaveIncomingShipment(shipmentGuid, false, dsSave, Common.UpdateFunction.UnitTests);
		}

	}
}
#endif


